# https://www.instana.com/blog/stans-robot-shop-sample-microservice-application/
# https://github.com/instana/robot-shop

import subprocess

from pyparsing import CaselessLiteral

from codeable_detectors.docker.docker_detectors import DockerEnvStatements, DockerMySQLDBImageBasedComponent
from codeable_detectors.dockerCompose.docker_compose_detectors import DockerComposeRabbitMQImageBasedComponent
from codeable_detectors.go.go_amqp_detectors import GoAMQPService, GoAMQPMessageConsumerLink
from codeable_detectors.go.go_instana_detectors import GoInstanaLink
from codeable_detectors.java.java_apache_http_client_detectors import JavaApacheHTTPLink
from codeable_detectors.java.java_jdbc_detectors import JavaMySQLJDBCLink
from codeable_detectors.java.java_spark_detectors import JavaSparkService
from codeable_detectors.js.jsExpressDetectors import JSExpressService
from codeable_detectors.js.jsInstanaDetectors import JSInstanaComponent, JSInstanaLink
from codeable_detectors.js.jsMongoDetectors import JSMongoDBComponent
from codeable_detectors.js.jsPrometheusDetectors import JSPrometheusMonitor
from codeable_detectors.js.jsRequestDetectors import JSRequestLink
from codeable_detectors.js.nodeJSRedisAPIDetectors import NodeJSRedisDBComponent, NodeJSRedisDBLink
from codeable_detectors.nginx.nginxDetectors import NginxAPIGateway, NginxLocationToServiceLink, \
    NginxLocationsToRESTAPIClient, NginxLocationsToWebUIClient
from codeable_detectors.php.phpCurlDetectors import PHPCurlLink
from codeable_detectors.php.phpDatabaseConnectionDetectors import PHPMySQLPDOLink
from codeable_detectors.php.phpRestfulDetectors import PHPRestfulService
from codeable_detectors.project import Project
from codeable_detectors.python.flaskDetectors import PythonFlaskRestfulService
from codeable_detectors.python.pythonAMQPDetectors import PythonPikaAMQPMessageProducerLink
from codeable_detectors.python.pythonOpenTracingDetectors import PythonOpenTracingLink
from codeable_detectors.python.pythonPrometheusDetectors import PythonPrometheusMonitor
from codeable_detectors.python.pythonRequestsDetectors import PythonRequestsCallToExternalComponent, PythonRequestsLink

project = Project("Microservice Based Robot Shop", "../systems/robot-shop-master")
project.add_env_vars(DockerEnvStatements(), "./web")

cart = project.add_component(JSExpressService(), "./cart")
catalogue = project.add_component(JSExpressService(), "./catalogue")
dispatch = project.add_component(GoAMQPService(), "./dispatch")
payment = project.add_component(PythonFlaskRestfulService(), "./payment")
ratings = project.add_component(PHPRestfulService(), "./ratings")
shipping = project.add_component(JavaSparkService(), "./shipping")
user = project.add_component(JSExpressService(), "./user")

apiGateway = project.add_component(NginxAPIGateway(), "./web")
project.add_links(NginxLocationToServiceLink(), apiGateway,
                  [catalogue, user, cart, shipping, payment, ratings])

webClient = project.add_component(NginxLocationsToRESTAPIClient(), "./web")
project.add_link(webClient, webClient, apiGateway)

webUI = project.add_component(NginxLocationsToWebUIClient(), "./web")
project.add_link(webUI, webUI, apiGateway)

project.add_link(JSRequestLink(), cart, catalogue)

cartAndAnonymousUserCountDB = project.add_component(NodeJSRedisDBComponent(), "./cart",
                                                    name="Cart and Anonymous User Count DB")
project.add_link(cartAndAnonymousUserCountDB, cart, cartAndAnonymousUserCountDB)
project.add_link(NodeJSRedisDBLink(), user, cartAndAnonymousUserCountDB)

cartPrometheus = project.add_component(JSPrometheusMonitor(), "./cart")
project.add_link(cartPrometheus, cart, cartPrometheus)

catalogueUsersOrdersDB = project.add_component(JSMongoDBComponent(), "./catalogue",
                                               name='Catalogue Users Orders DB')
project.add_link(catalogueUsersOrdersDB, catalogue, catalogueUsersOrdersDB)
project.add_link(catalogueUsersOrdersDB, user, catalogueUsersOrdersDB)

paymentGateway = project.add_component(PythonRequestsCallToExternalComponent([CaselessLiteral("PAYMENT"),
                                                                              CaselessLiteral("GATEWAY")]),
                                       "./payment", name='Paypal Payment Gateway')
project.add_link(paymentGateway, payment, paymentGateway)

project.add_link(PythonRequestsLink(), payment, user)
project.add_link(PythonRequestsLink(), payment, cart)

paymentPrometheus = project.add_component(PythonPrometheusMonitor(), "./payment")
project.add_link(paymentPrometheus, payment, paymentPrometheus)

# we just use one of the instana evidences for adding the Instana external component
instana = project.add_component(JSInstanaComponent(), "./cart")
project.add_link(instana, cart, instana)
# but now need to check evidence each other link to the instance component separately 
project.add_link(JSInstanaLink(), catalogue, instana)
project.add_link(GoInstanaLink(), dispatch, instana)
project.add_link(PythonOpenTracingLink(), payment, instana)
project.add_link(JSInstanaLink(), user, instana)

rabbitMQ = project.add_component(DockerComposeRabbitMQImageBasedComponent(), "./dispatch")
project.add_link(PythonPikaAMQPMessageProducerLink(), payment, rabbitMQ, channel="orders")
project.add_link(GoAMQPMessageConsumerLink(), dispatch, rabbitMQ, channel="orders")

ratingsAndCitiesDB = project.add_component(DockerMySQLDBImageBasedComponent(), "./mysql",
                                           name="Ratings and Shipping Cities DB")
project.add_link(PHPMySQLPDOLink(), ratings, ratingsAndCitiesDB)

project.add_link(PHPCurlLink(), ratings, catalogue)

project.add_link(JavaApacheHTTPLink(), shipping, cart)

project.add_link(JavaMySQLJDBCLink(), shipping, ratingsAndCitiesDB)

if project.failed_evidences.have_failed():
    print("*** FAILED EVIDENCES / MODEL ELEMENTS NOT ADDED ***\n")
    no = 0
    for failed in project.failed_evidences.get():
        no = no + 1
        print(str(no) + ": " + str(failed))
    print("\n*** \n")

# print("*** RESULTING MODEL ***\n")
# print(project.genModel)
# print("*** \n")

project.save_as_file("genmodel.py")
subprocess.call(['python', 'generateAll.py'])
